<?php
/**
 * Chat Gateway Endpoint
 * Main endpoint for chat interactions with AI and live agent handoff
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth_middleware.php';
require_once __DIR__ . '/../includes/ai_service.php';
require_once __DIR__ . '/../includes/pusher_config.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Get vendor ID from token (for widget, token might be in query param or header)
    $vendorId = null;
    $chatbotId = null;
    
    // Try to get from Authorization header or cookie
    try {
        $vendorData = AuthMiddleware::authenticate();
        $vendorId = $vendorData['vendor_id'];
    } catch (Exception $e) {
        // For widget, vendor_id might be passed directly
        // This is less secure but needed for public widget access
        // In production, consider using API keys or signed tokens
    }
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (empty($input['message'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Message is required']);
        exit;
    }
    
    // Get vendor_id and chatbot_id from input (for widget access)
    if (empty($vendorId)) {
        $vendorId = (int) ($input['vendor_id'] ?? 0);
    }
    $chatbotId = (int) ($input['chatbot_id'] ?? 0);
    $sessionId = $input['session_id'] ?? uniqid('session_', true);
    $userMessage = trim($input['message']);
    
    if (empty($vendorId) || empty($chatbotId)) {
        http_response_code(400);
        echo json_encode(['error' => 'vendor_id and chatbot_id are required']);
        exit;
    }
    
    $db = getDB();
    
    // Verify chatbot belongs to vendor
    $stmt = $db->prepare("
        SELECT c.id, c.system_prompt, c.live_agent_status, v.plan_type
        FROM chatbots c
        INNER JOIN vendors v ON c.vendor_id = v.id
        WHERE c.id = ? AND c.vendor_id = ?
    ");
    $stmt->execute([$chatbotId, $vendorId]);
    $chatbot = $stmt->fetch();
    
    if (!$chatbot) {
        http_response_code(404);
        echo json_encode(['error' => 'Chatbot not found']);
        exit;
    }
    
    // Check if session is escalated (live agent)
    $stmt = $db->prepare("
        SELECT status
        FROM active_sessions
        WHERE session_id = ? AND vendor_id = ?
    ");
    $stmt->execute([$sessionId, $vendorId]);
    $activeSession = $stmt->fetch();
    
    if ($activeSession && $activeSession['status'] === 'escalated') {
        // Session is escalated - store message for live agent
        $stmt = $db->prepare("
            INSERT INTO chat_logs (vendor_id, chatbot_id, session_id, role, message)
            VALUES (?, ?, ?, 'user', ?)
        ");
        $stmt->execute([$vendorId, $chatbotId, $sessionId, $userMessage]);
        
        // Notify vendor via Pusher
        try {
            $pusher = PusherConfig::getInstance();
            $pusher->trigger(
                "vendor-{$vendorId}",
                "chat-update",
                [
                    'session_id' => $sessionId,
                    'role' => 'user',
                    'message' => $userMessage,
                    'timestamp' => date('Y-m-d H:i:s')
                ]
            );
        } catch (Exception $e) {
            error_log("Pusher error: " . $e->getMessage());
        }
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'message' => 'Message sent to live agent',
            'session_id' => $sessionId,
            'escalated' => true
        ]);
        exit;
    }
    
    // Process with AI
    $_POST['session_id'] = $sessionId; // For ai_service.php
    $response = AIService::getChatResponse($userMessage, $vendorId, $chatbotId);
    
    $confidence = $response['confidence'] ?? 0.0;
    $escalationAvailable = false;
    
    // Check if escalation is available (Pro plan only)
    if ($chatbot['plan_type'] === 'pro' && $chatbot['live_agent_status'] === 'enabled') {
        // Show escalation option if confidence is low
        if ($confidence < 0.7) {
            $escalationAvailable = true;
        }
    }
    
    // Update or create active session
    $stmt = $db->prepare("
        INSERT INTO active_sessions (vendor_id, chatbot_id, session_id, status)
        VALUES (?, ?, ?, 'active')
        ON DUPLICATE KEY UPDATE updated_at = CURRENT_TIMESTAMP
    ");
    $stmt->execute([$vendorId, $chatbotId, $sessionId]);
    
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => $response['message'],
        'session_id' => $response['session_id'],
        'confidence' => $confidence,
        'escalation_available' => $escalationAvailable
    ]);
    
} catch (Exception $e) {
    error_log("Chat gateway error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
