<?php
/**
 * Delete Knowledge Entry Endpoint
 * Removes knowledge entry from database and Qdrant
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth_middleware.php';
require_once __DIR__ . '/../includes/env_loader.php';

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Authenticate vendor
    $vendorId = AuthMiddleware::getVendorId();
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Knowledge entry ID is required']);
        exit;
    }
    
    $knowledgeId = (int) $input['id'];
    
    $db = getDB();
    
    // Get knowledge entry and verify ownership
    $stmt = $db->prepare("
        SELECT id, qdrant_point_id
        FROM knowledge_base
        WHERE id = ? AND vendor_id = ?
    ");
    $stmt->execute([$knowledgeId, $vendorId]);
    $entry = $stmt->fetch();
    
    if (!$entry) {
        http_response_code(404);
        echo json_encode(['error' => 'Knowledge entry not found']);
        exit;
    }
    
    // Delete from Qdrant if point ID exists
    if (!empty($entry['qdrant_point_id'])) {
        try {
            $apiKey = getenv('QDRANT_API_KEY');
            $clusterUrl = getenv('QDRANT_CLUSTER_URL');
            $collectionName = getenv('QDRANT_COLLECTION_NAME') ?: 'chatbot_knowledge';
            
            $client = new Client([
                'base_uri' => rtrim($clusterUrl, '/') . '/',
                'headers' => [
                    'api-key' => $apiKey,
                    'Content-Type' => 'application/json'
                ],
                'timeout' => 30
            ]);
            
            $client->post("collections/{$collectionName}/points/delete", [
                'json' => [
                    'points' => [$entry['qdrant_point_id']]
                ]
            ]);
        } catch (Exception $e) {
            error_log("Failed to delete from Qdrant: " . $e->getMessage());
            // Continue with database deletion even if Qdrant deletion fails
        }
    }
    
    // Delete from database
    $stmt = $db->prepare("DELETE FROM knowledge_base WHERE id = ? AND vendor_id = ?");
    $stmt->execute([$knowledgeId, $vendorId]);
    
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Knowledge entry deleted successfully'
    ]);
    
} catch (Exception $e) {
    error_log("Delete knowledge error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
