<?php
/**
 * Escalate Chat Endpoint
 * Handles escalation to live agent
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/pusher_config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['vendor_id']) || empty($input['chatbot_id']) || empty($input['session_id'])) {
        http_response_code(400);
        echo json_encode(['error' => 'vendor_id, chatbot_id, and session_id are required']);
        exit;
    }
    
    $vendorId = (int) $input['vendor_id'];
    $chatbotId = (int) $input['chatbot_id'];
    $sessionId = $input['session_id'];
    
    $db = getDB();
    
    // Verify chatbot belongs to vendor
    $stmt = $db->prepare("
        SELECT c.id, v.plan_type, c.live_agent_status
        FROM chatbots c
        INNER JOIN vendors v ON c.vendor_id = v.id
        WHERE c.id = ? AND c.vendor_id = ?
    ");
    $stmt->execute([$chatbotId, $vendorId]);
    $chatbot = $stmt->fetch();
    
    if (!$chatbot) {
        http_response_code(404);
        echo json_encode(['error' => 'Chatbot not found']);
        exit;
    }
    
    // Check if Pro plan and live agent is enabled
    if ($chatbot['plan_type'] !== 'pro' || $chatbot['live_agent_status'] !== 'enabled') {
        http_response_code(403);
        echo json_encode(['error' => 'Live agent feature is not available']);
        exit;
    }
    
    // Update active session status
    $stmt = $db->prepare("
        INSERT INTO active_sessions (vendor_id, chatbot_id, session_id, status)
        VALUES (?, ?, ?, 'escalated')
        ON DUPLICATE KEY UPDATE status = 'escalated', updated_at = CURRENT_TIMESTAMP
    ");
    $stmt->execute([$vendorId, $chatbotId, $sessionId]);
    
    // Notify vendor via Pusher
    try {
        $pusher = PusherConfig::getInstance();
        $pusher->trigger(
            "vendor-{$vendorId}",
            "chat-escalated",
            [
                'session_id' => $sessionId,
                'chatbot_id' => $chatbotId,
                'timestamp' => date('Y-m-d H:i:s')
            ]
        );
    } catch (Exception $e) {
        error_log("Pusher error: " . $e->getMessage());
    }
    
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Chat escalated to live agent'
    ]);
    
} catch (Exception $e) {
    error_log("Escalate chat error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
