<?php
/**
 * Vendor Signup Endpoint
 * Handles vendor registration with password hashing and trial period setup
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/jwt_helper.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (empty($input['business_name']) || empty($input['email']) || empty($input['password'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing required fields: business_name, email, password']);
        exit;
    }
    
    $businessName = trim($input['business_name']);
    $email = filter_var(trim($input['email']), FILTER_SANITIZE_EMAIL);
    $password = $input['password'];
    
    // Validate email format
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid email format']);
        exit;
    }
    
    // Validate password strength (minimum 8 characters)
    if (strlen($password) < 8) {
        http_response_code(400);
        echo json_encode(['error' => 'Password must be at least 8 characters long']);
        exit;
    }
    
    $db = getDB();
    
    // Check if email already exists
    $stmt = $db->prepare("SELECT id FROM vendors WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        http_response_code(409);
        echo json_encode(['error' => 'Email already registered']);
        exit;
    }
    
    // Hash password
    $passwordHash = password_hash($password, PASSWORD_BCRYPT);
    
    // Calculate trial end date (30 days from now)
    $trialEndsAt = date('Y-m-d H:i:s', strtotime('+30 days'));
    
    // Insert vendor
    $stmt = $db->prepare("
        INSERT INTO vendors (business_name, email, password_hash, plan_type, status, trial_ends_at)
        VALUES (?, ?, ?, 'starter', 'trial', ?)
    ");
    
    $stmt->execute([$businessName, $email, $passwordHash, $trialEndsAt]);
    $vendorId = $db->lastInsertId();
    
    // Create default chatbot for the vendor
    $stmt = $db->prepare("
        INSERT INTO chatbots (vendor_id, bot_name, welcome_message, system_prompt)
        VALUES (?, 'My Chatbot', 'Hello! How can I help you today?', 'You are a helpful assistant. Answer questions based on the provided context.')
    ");
    $stmt->execute([$vendorId]);
    
    // Generate JWT token
    $token = JWTHelper::encode([
        'vendor_id' => $vendorId,
        'email' => $email,
        'plan_type' => 'starter',
        'trial_ends_at' => $trialEndsAt
    ]);
    
    // Set token as HTTP-only cookie
    JWTHelper::setTokenCookie($token);
    
    // Return success response
    http_response_code(201);
    echo json_encode([
        'success' => true,
        'message' => 'Vendor registered successfully',
        'vendor' => [
            'id' => $vendorId,
            'business_name' => $businessName,
            'email' => $email,
            'plan_type' => 'starter',
            'trial_ends_at' => $trialEndsAt
        ],
        'token' => $token
    ]);
    
} catch (PDOException $e) {
    error_log("Signup error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Database error occurred']);
} catch (Exception $e) {
    error_log("Signup error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
