<?php
/**
 * Train Bot Endpoint
 * Saves content to knowledge base and ingests into Qdrant
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth_middleware.php';
require_once __DIR__ . '/../includes/ai_service.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Authenticate vendor
    $vendorId = AuthMiddleware::getVendorId();
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    if (empty($input['content'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Content is required']);
        exit;
    }
    
    $content = trim($input['content']);
    $contentType = $input['content_type'] ?? 'text';
    
    // Validate content type
    if (!in_array($contentType, ['faq', 'doc', 'text'])) {
        $contentType = 'text';
    }
    
    // Ingest document into Qdrant and store in knowledge base
    $pointIds = AIService::ingestDocument($content, $vendorId, $contentType);
    
    if (empty($pointIds)) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to ingest document']);
        exit;
    }
    
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => 'Bot trained successfully with ' . count($pointIds) . ' knowledge chunks',
        'chunks_created' => count($pointIds)
    ]);
    
} catch (Exception $e) {
    error_log("Train bot error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
