<?php
/**
 * Training Page
 * Knowledge base management interface for vendors
 */

session_start();
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/auth_middleware.php';

// Authenticate user
$vendorData = AuthMiddleware::authenticate();
$vendorId = $vendorData['vendor_id'];

// Get existing knowledge entries
$db = getDB();
$stmt = $db->prepare("
    SELECT id, content_type, raw_content, created_at
    FROM knowledge_base
    WHERE vendor_id = ?
    ORDER BY created_at DESC
");
$stmt->execute([$vendorId]);
$knowledgeEntries = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Train Your Bot - AI Chatbot SaaS</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: #f5f5f5;
            color: #333;
        }
        .header {
            background: #fff;
            border-bottom: 1px solid #e0e0e0;
            padding: 1rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .header h1 {
            font-size: 1.5rem;
            color: #2563eb;
        }
        .nav-link {
            color: #2563eb;
            text-decoration: none;
            margin-right: 1rem;
        }
        .container {
            max-width: 1200px;
            margin: 2rem auto;
            padding: 0 2rem;
        }
        .card {
            background: #fff;
            padding: 2rem;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            margin-bottom: 2rem;
        }
        .card h2 {
            margin-bottom: 1.5rem;
            font-size: 1.5rem;
        }
        .form-group {
            margin-bottom: 1.5rem;
        }
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: #374151;
        }
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #d1d5db;
            border-radius: 6px;
            font-size: 1rem;
            font-family: inherit;
        }
        .form-group textarea {
            min-height: 200px;
            resize: vertical;
        }
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #2563eb;
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 6px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
        }
        .btn-primary {
            background: #2563eb;
            color: white;
        }
        .btn-primary:hover {
            background: #1d4ed8;
        }
        .btn-danger {
            background: #ef4444;
            color: white;
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
        }
        .btn-danger:hover {
            background: #dc2626;
        }
        .knowledge-list {
            margin-top: 2rem;
        }
        .knowledge-item {
            padding: 1rem;
            border: 1px solid #e5e7eb;
            border-radius: 6px;
            margin-bottom: 1rem;
            background: #f9fafb;
        }
        .knowledge-item-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.5rem;
        }
        .knowledge-item-type {
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
        }
        .type-faq {
            background: #dbeafe;
            color: #1e40af;
        }
        .type-doc {
            background: #fce7f3;
            color: #9f1239;
        }
        .type-text {
            background: #f3f4f6;
            color: #374151;
        }
        .knowledge-item-content {
            color: #6b7280;
            font-size: 0.875rem;
            margin-bottom: 0.5rem;
            max-height: 100px;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        .knowledge-item-date {
            color: #9ca3af;
            font-size: 0.75rem;
        }
        .alert {
            padding: 1rem;
            border-radius: 6px;
            margin-bottom: 1rem;
        }
        .alert-success {
            background: #d1fae5;
            border: 1px solid #10b981;
            color: #065f46;
        }
        .alert-error {
            background: #fee2e2;
            border: 1px solid #ef4444;
            color: #991b1b;
        }
        .loading {
            display: none;
            text-align: center;
            padding: 1rem;
            color: #6b7280;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>AI Chatbot SaaS</h1>
        <div>
            <a href="index.php" class="nav-link">← Dashboard</a>
        </div>
    </div>

    <div class="container">
        <div class="card">
            <h2>Train Your Bot</h2>
            <div id="alert-container"></div>
            <div class="loading" id="loading">Training your bot... Please wait.</div>
            
            <form id="training-form">
                <div class="form-group">
                    <label for="content_type">Content Type</label>
                    <select id="content_type" name="content_type" required>
                        <option value="faq">FAQ</option>
                        <option value="doc">Document</option>
                        <option value="text" selected>General Text</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="content">Content</label>
                    <textarea 
                        id="content" 
                        name="content" 
                        placeholder="Enter FAQs, documents, or any text content to train your chatbot. The bot will use this information to answer questions from your customers."
                        required
                    ></textarea>
                </div>
                
                <button type="submit" class="btn btn-primary">Save & Train Bot</button>
            </form>
        </div>

        <div class="card">
            <h2>Knowledge Base (<?php echo count($knowledgeEntries); ?> entries)</h2>
            <div class="knowledge-list">
                <?php if (empty($knowledgeEntries)): ?>
                    <p style="color: #6b7280; text-align: center; padding: 2rem;">No knowledge entries yet. Add content above to train your bot.</p>
                <?php else: ?>
                    <?php foreach ($knowledgeEntries as $entry): ?>
                    <div class="knowledge-item">
                        <div class="knowledge-item-header">
                            <span class="knowledge-item-type type-<?php echo $entry['content_type']; ?>">
                                <?php echo strtoupper($entry['content_type']); ?>
                            </span>
                            <button 
                                class="btn btn-danger" 
                                onclick="deleteKnowledge(<?php echo $entry['id']; ?>)"
                            >
                                Delete
                            </button>
                        </div>
                        <div class="knowledge-item-content">
                            <?php echo htmlspecialchars(substr($entry['raw_content'], 0, 200)); ?>
                            <?php if (strlen($entry['raw_content']) > 200): ?>...<?php endif; ?>
                        </div>
                        <div class="knowledge-item-date">
                            Added: <?php echo date('F j, Y g:i A', strtotime($entry['created_at'])); ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
        document.getElementById('training-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = {
                content_type: document.getElementById('content_type').value,
                content: document.getElementById('content').value
            };
            
            const loading = document.getElementById('loading');
            const alertContainer = document.getElementById('alert-container');
            loading.style.display = 'block';
            alertContainer.innerHTML = '';
            
            try {
                const response = await fetch('../api/train_bot.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(formData)
                });
                
                const data = await response.json();
                loading.style.display = 'none';
                
                if (response.ok && data.success) {
                    alertContainer.innerHTML = '<div class="alert alert-success">✓ ' + data.message + '</div>';
                    document.getElementById('content').value = '';
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    alertContainer.innerHTML = '<div class="alert alert-error">✗ ' + (data.error || 'Failed to train bot') + '</div>';
                }
            } catch (error) {
                loading.style.display = 'none';
                alertContainer.innerHTML = '<div class="alert alert-error">✗ Network error: ' + error.message + '</div>';
            }
        });
        
        async function deleteKnowledge(id) {
            if (!confirm('Are you sure you want to delete this knowledge entry?')) {
                return;
            }
            
            try {
                const response = await fetch('../api/delete_knowledge.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ id: id })
                });
                
                const data = await response.json();
                
                if (response.ok && data.success) {
                    location.reload();
                } else {
                    alert('Error: ' + (data.error || 'Failed to delete'));
                }
            } catch (error) {
                alert('Network error: ' + error.message);
            }
        }
    </script>
</body>
</html>
